<?php
/**
 * Collects leads and subscribes to Infusionsoft (Infusionsoft)
 *
 * @package Convert Pro Addon
 */

/**
 * Helper class for the Infusionsoft (Keap) API.
 *
 * @since 1.0.0
 */
final class CPRO_Service_Infusionsoft extends CPRO_Service {

	/**
	 * The ID for this service.
	 *
	 * @since 1.0.0
	 * @var string $id
	 */
	public $id = 'infusionsoft';

	/**
	 * Default Custom field array.
	 *
	 * @since 1.0.0
	 * @var string $id
	 */
	public static $mapping_fields = array( 'given_name', 'family_name', 'email', 'phone_number', 'address', 'zip_code', 'website' );

	/**
	 * Store API instance
	 *
	 * @since 1.0.0
	 * @var object $api_instance
	 * @access private
	 */
	private $api_instance = null;

	/**
	 * Get an instance of the API.
	 *
	 * @since 1.0.0
	 * @param string $access_token A valid access token.
	 * @return object The API instance.
	 */
	public function get_api( $access_token ) {

		if ( $this->api_instance ) {
			return $this->api_instance;
		}

		if ( file_exists( CP_SERVICES_BASE_DIR . 'includes/vendor/infusionsoft/keap.php' ) ) {
			require_once CP_SERVICES_BASE_DIR . 'includes/vendor/infusionsoft/keap.php';
		}

		if ( class_exists( 'CPRO_Keap' ) ) {
			$this->api_instance = new CPRO_Keap( $access_token );
		}

		return $this->api_instance;
	}

	/**
	 * Test the API connection.
	 *
	 * @since 1.0.0
	 * @param array $fields A valid access token.
	 * @return array {
	 *      @type bool|string $error The error message or false if no error.
	 *      @type array $data An array of data used to make the connection.
	 * }
	 */
	public function connect( $fields = array() ) {
		$response = array(
			'error' => false,
			'data'  => array(),
		);

		// Make sure we have an access token.
		if ( ! isset( $fields['access_token'] ) || empty( $fields['access_token'] ) || ! is_string( $fields['access_token'] ) ) {
			$response['error'] = __( 'Error: You must provide an access token.', 'convertpro-addon' );
		} else {
			// Try to connect and store the connection data.
			$api = $this->get_api( sanitize_text_field( $fields['access_token'] ) );

			try {
				$connected = $api->connect();

				if ( false !== $connected['error'] ) {
					$response['error'] = $connected['error'];
				}

				$response['data'] = array(
					'access_token' => sanitize_text_field( $fields['access_token'] ),
				);
			} catch ( Exception $e ) {
				$response['error'] = $e->getMessage();
			}
		}
		return $response;
	}

	/**
	 * Renders the markup for the connection settings.
	 *
	 * @since 1.0.0
	 * @return string The connection settings markup.
	 */
	public function render_connect_settings() {
		ob_start();

		ConvertPlugHelper::render_input_html(
			'access_token',
			array(
				'class' => '',
				'type'  => 'text',
				'label' => __( 'Access Token', 'convertpro-addon' ),
				'desc'  => __( '<p>Your Access Token can be obtained from your Keap (Infusionsoft) under Account > Settings > API > Personal Access Tokens.</p>', 'convertpro-addon' ),
			)
		);
		return ob_get_clean();
	}

	/**
	 * Returns the access_token in array format
	 *
	 * @since 1.0.0
	 * @param string $auth_meta A valid access token.
	 * @return array Array of access_token
	 */
	public function render_auth_meta( $auth_meta ) {
		return array(
			'access_token' => $auth_meta['access_token'],
		);
	}

	/**
	 * Render the markup for service specific fields.
	 *
	 * @since 1.0.0
	 * @param string $account The name of the saved account.
	 * @param object $post_data Posted data.
	 * @return array {
	 *      @type bool|string $error The error message or false if no error.
	 *      @type string $html The field markup.
	 *      @type array $mapping_fields The field mapping array for keap.
	 * }
	 */
	public function render_fields( $account, $post_data ) {
		$account_data = ConvertPlugServices::get_account_data( $account );

		$api                 = $this->get_api( $account_data['access_token'] );
		$response            = array(
			'error'          => false,
			'html'           => '',
			'mapping_fields' => self::$mapping_fields,
		);
		$post_data['isEdit'] = ( isset( $post_data['isEdit'] ) ) ? $post_data['isEdit'] : null;

		try {
			$resp = $api->getList();

			if ( false === $resp['error'] ) {

				if ( ! empty( $resp['lists'] ) ) {

					$lists             = $resp['lists'];
					$response['html'] .= $this->render_list_field( $lists, $post_data );
				} else {
					$response['error'] = __( 'No list added yet.', 'convertpro-addon' );
				}
			} else {
				$response['error'] = $resp['error'];
			}
		} catch ( Exception $e ) {
			$response['error'] = $e->getMessage();
		}
		return $response;
	}

	/**
	 * Render markup for the list field.
	 *
	 * @since 1.0.0
	 * @param array $lists List data from the API.
	 * @param array $settings Posted data.
	 * @return string The markup for the list field.
	 * @access private
	 */
	private function render_list_field( $lists, $settings ) {

		$list_options = array(
			'-1' => __( 'Choose...', 'convertpro-addon' ),
		);
		$default      = '';

		foreach ( $lists as $list ) {
			$list_options[ $list->id ] = $list->name;
		}

		if ( isset( $settings['isEdit'] ) && $settings['isEdit'] ) {
			$default = ( isset( $settings['default'] ) ) ? ( ( isset( $settings['default']['infusionsoft_tags'] ) ) ? $settings['default']['infusionsoft_tags'] : '' ) : '';
		}

		ob_start();

		ConvertPlugHelper::render_input_html(
			'infusionsoft_tags',
			array(
				'class'   => '',
				'type'    => 'multi-select',
				'label'   => __( 'Select Tags', 'convertpro-addon' ),
				'help'    => '',
				'default' => $default,
				'options' => $list_options,
			)
		);

		return ob_get_clean();

	}

	/**
	 * Mapping fields.
	 *
	 * @since 1.0.0
	 */
	public function render_mapping() {
		return self::$mapping_fields;
	}

	/**
	 * Subscribe an email address to Keap.
	 *
	 * @since 1.0.0
	 * @param object $settings A module settings object.
	 * @param string $email The email to subscribe.
	 * @return array {
	 *      @type bool|string $error The error message or false if no error.
	 * }
	 */
	public function subscribe( $settings, $email ) {

		$account_data = ConvertPlugServices::get_account_data( $settings['api_connection'] );

		$response = array(
			'error' => false,
		);

		if ( ! $account_data ) {
			$response['error'] = __( 'There was an error subscribing to Keap (Infusionsoft)! The account is no longer connected.', 'convertpro-addon' );
		} else {

			$api = $this->get_api( $account_data['access_token'] );

			$data               = array();
			$data['properties'] = array();
			$fields             = array();
			$custom_fields      = array();
			$cust_fields        = array();

			foreach ( $settings['param'] as $key => $p ) {

				if ( 'email' !== $key && 'date' !== $key ) {
					if ( isset( $settings['meta'][ $key ] ) ) {
						if ( 'custom_field' !== $settings['meta'][ $key ] ) {

							$fields[ $settings['meta'][ $key ] ] = $p;

						} else {

							$fields[ $settings['meta'][ $key . '-input' ] ] = $p;
							$custom_fields                                  = $fields;
						}
					}
				}
			}

			if ( ! empty( $fields ) ) {
				$data['properties'] = $fields;
			}

			try {
				$email      = sanitize_email( $email );
				$subscribed = $api->subscribe( $settings['infusionsoft_tags'], $email, $data );

				if ( false !== $subscribed['error'] ) {
					$response['error'] = $subscribed['error'];
				}
			} catch ( Exception $e ) {
				$response['error'] = $e->getMessage();
			}
		}

		return $response;
	}
}
