<?php
/**
 * Collects leads and subscribe to AWeber
 *
 * @package Convert Pro Addon
 * @author Brainstorm Force
 */

/**
 * Helper class for the AWeber API.
 *
 * @since 1.0.0
 */
final class CPRO_Service_AWeber extends CPRO_Service {

	/**
	 * The ID for this service.
	 *
	 * @since 1.0.0
	 * @var string $id
	 */
	public $id = 'aweber';

	/**
	 * Default Custom field array.
	 * This is predefined custom fields array that AWeber
	 * has already defined. When AWeber releases the new
	 * set of fields, we need to update this array.
	 *
	 * @since 1.0.0
	 * @var string $id
	 */
	public static $mapping_fields = array( 'name' );

	/**
	 * Store API instance
	 *
	 * @since 1.0.0
	 * @var object $api_instance
	 * @access private
	 */
	private $api_instance = null;

	/**
	 * Store Old API instance
	 *
	 * @since 1.0.0
	 * @var object $oldapi_instance
	 * @access private
	 */
	private $oldapi_instance = null;

	/**
	 * Get an instance of the API.
	 *
	 * @since 1.0.0
	 * @param string $auth_code A valid API key.
	 * @return object The API instance.
	 */
	public function get_api( $auth_code ) {
		if ( $this->api_instance ) {
			return $this->api_instance;
		}
		if ( ! class_exists( 'AWeberAPI' ) ) {
			require_once CP_SERVICES_BASE_DIR . 'includes/vendor/aweber/aweber_api.php';
		}

		$this->api_instance = $this->getAWeberOAuth2API( $auth_code );

		return $this->api_instance;
	}

	/**
	 * Test the API connection.
	 *
	 * @since 1.0.0
	 * @param array $fields A valid API key.
	 * @return array{
	 *      @type bool|string $error The error message or false if no error.
	 *      @type array $data An array of data used to make the connection.
	 * }
	 */
	public function connect( $fields = array() ) {

		$response            = array(
			'error' => false,
			'data'  => array(),
		);
		$access_token        = '';
		$access_token_secret = '';

		// Make sure we have an authorization code.
		if ( ! isset( $fields['auth_code'] ) || empty( $fields['auth_code'] ) ) {
			$response['error'] = __( 'Error: You must provide an Authorization Code.', 'convertpro-addon' );
		} else {

			// Try to connect and store the connection data.
			$api = $this->get_api( $fields['auth_code'] );

			$token_data = $api->generateAccessToken( $fields['auth_code'], $fields['service_account'] );

			if ( isset( $token_data['error'] ) ) {
				// Check the error is because of the In-correct authorization code was entered.
				if ( '400' === $token_data['status'] && 'invalid_request' === $token_data['error'] ) {
					$response['error'] = __( 'Error: The Authorization Code is invalid. Please enter a valid Authorization Code.', 'convertpro-addon' );
				} else {
					$response['error'] = __( 'Error: There was an error connecting to AWeber. Please try again.', 'convertpro-addon' );
				}
			}

			$response['data'] = array(
				'auth_code'     => isset( $fields['auth_code'] ) ? $fields['auth_code'] : '',
				'access_token'  => isset( $token_data['access_token'] ) ? $token_data['access_token'] : '',
				'refresh_token' => isset( $token_data['refresh_token'] ) ? $token_data['refresh_token'] : '',
				'expires_in'    => isset( $token_data['expires_in'] ) ? $token_data['expires_in'] : '',
				'token_type'    => isset( $token_data['token_type'] ) ? $token_data['token_type'] : '',
				'expires_on'    => isset( $token_data['expires_on'] ) ? $token_data['expires_on'] : '',
			);

			update_option( '_cp_v2_aweber_temp', $response['data'] );

		}

		return $response;
	}

	/**
	 * Renders the markup for the connection settings.
	 *
	 * @since 1.0.0
	 * @return string The connection settings markup.
	 */
	public function render_connect_settings() {

		$aweber = $this->getAWeberOAuth2API();
		// Generate the Authorization URL.
		$authorizeUrl = $aweber->getAuthorizeUrl();

		ob_start();

		ConvertPlugHelper::render_input_html(
			'auth_code',
			array(
				'class' => '',
				'type'  => 'text',
				'label' => __( 'Authorization Code', 'convertpro-addon' ),
				/* translators: %s Link */
				'desc'  => sprintf( __( '<p>Please register this website with AWeber to get your Authorization Code. <a%s rel="noopener">Register Now!</a></p>', 'convertpro-addon' ), ' href="' . $authorizeUrl . '" target="_blank"' ),
			)
		);

		return ob_get_clean();
	}

	/**
	 * Get AWeber 2.0 api.
	 *
	 * @since 1.5.9
	 * @param string $auth_code authentication code entered by user.
	 * @return class instance.
	 */
	public function getAWeberOAuth2API( $auth_code = '' ) {

		if ( ! class_exists( 'AWeberAPI' ) ) {
			require_once CP_SERVICES_BASE_DIR . 'includes/vendor/aweber/aweber_api.php';
		}

		$oauth2TokensOptions = get_option( '_cp_v2_aweber_temp' );

		if ( isset( $oauth2TokensOptions['access_token'] ) && ( isset( $auth_code ) && $auth_code === $oauth2TokensOptions['auth_code'] ) ) {

			return new AWeberOAuth2API(
				$oauth2TokensOptions['access_token'],
				$oauth2TokensOptions['refresh_token'],
				$oauth2TokensOptions['expires_on']
			);
		} elseif ( isset( $auth_code ) && is_array( $oauth2TokensOptions ) && isset( $oauth2TokensOptions['auth_code'] ) && $auth_code !== $oauth2TokensOptions['auth_code'] ) {

			$accounts_list = get_option( 'cp_aweber_account_list', array() );

			if ( array_key_exists( $auth_code, $accounts_list ) ) {
				$account = $accounts_list[ $auth_code ];
				return new AWeberOAuth2API(
					$account['access_token'],
					$account['refresh_token'],
					$account['expires_on']
				);
			}
		}

		return new AWeberOAuth2API();
	}


	/**
	 * Returns the api_key in array format.
	 *
	 * @since 1.0.0
	 * @param string $auth_meta $api_key A valid API key.
	 * @return array Array of api_key.
	 */
	public function render_auth_meta( $auth_meta ) {

		$opt = get_option( '_cp_v2_aweber_temp' );

		return array(
			'auth_code'     => $auth_meta['auth_code'],
			'access_token'  => $opt['access_token'],
			'refresh_token' => $opt['refresh_token'],
			'expires_in'    => $opt['expires_in'],
			'token_type'    => $opt['token_type'],
			'expires_on'    => $opt['expires_on'],
		);
	}

	/**
	 * Render the markup for service specific fields.
	 *
	 * @since 1.0.0
	 * @param string $account The name of the saved account.
	 * @param object $post_data Posted data.
	 * @return array {
	 *      @type bool|string $error The error message or false if no error.
	 *      @type string $html The field markup.
	 *      @type array $mapping_fields The field mapping array for AWeber.
	 * }
	 */
	public function render_fields( $account, $post_data ) {

		$account_data        = ConvertPlugServices::get_account_data( $account );
		$authentication_code = $account_data['auth_code'];
		$response            = array(
			'error'          => false,
			'html'           => '',
			'mapping_fields' => self::$mapping_fields,
		);

		if ( $this->is_old_api_called( $authentication_code ) ) {
			$response['error'] = __( 'Error! Selected account is using deprecated API, to use latest remove and reconnect your account.', 'convertpro-addon' );

			return $response;
		}

		$api = $this->get_api( $authentication_code );

		$i           = 0;
		$collections = array();
		try {
			$account = $api->getAccount( $authentication_code );

			$lists = $account->loadFromUrl( '/accounts/' . $account->id . '/lists' );

			$collections[ $i ] = $lists->data['entries'];
			$temp_list         = $lists->data;
			while ( ! empty( $temp_list['entries'] ) ) {
				if ( array_key_exists( 'next_collection_link', $temp_list ) ) {
					$lists     = $account->loadFromUrl( $temp_list['next_collection_link'] );
					$temp_list = $lists->data;

					$collections[ $i++ ] = $temp_list['entries'];
				} else {
					$temp_list['entries'] = array();
				}
			}

			$response['html']  = $this->render_list_field( $collections, $post_data );
			$response['html'] .= $this->render_tags_field( $post_data );
		} catch ( AWeberException $e ) {
			$response['error'] = $e->getMessage();
		}

		return $response;
	}

	/**
	 * Render markup for the list field.
	 *
	 * @since 1.0.0
	 * @param array $lists List data from the API.
	 * @param array $settings Posted data.
	 * @return string The markup for the list field.
	 * @access private
	 */
	private function render_list_field( $lists, $settings ) {
		$default = '';

		if ( isset( $settings['isEdit'] ) && $settings['isEdit'] ) {
			$default = ( isset( $settings['default'] ) && isset( $settings['default']['aweber_list'] ) ) ? $settings['default']['aweber_list'] : '';
		}

		foreach ( $lists as $list ) {
			foreach ( $list as $value ) {
				$options[ $value['id'] ] = $value['name'];
			}
		}

		// Sort it into Alphabetical order.
		asort( $options );
		$options = array( '-1' => __( 'Choose...', 'convertpro-addon' ) ) + $options;

		ob_start();

		ConvertPlugHelper::render_input_html(
			'aweber_list',
			array(
				'class'   => '',
				'type'    => 'select',
				'label'   => __( 'Select a List', 'convertpro-addon' ),
				'help'    => '',
				'default' => $default,
				'options' => $options,
			)
		);

		return ob_get_clean();
	}

	/**
	 * Render markup for the tag field.
	 *
	 * @since 1.0.0
	 * @param array $settings Posted data.
	 * @return string The markup for the list field.
	 * @access private
	 */
	private function render_tags_field( $settings ) {

		if ( isset( $settings['isEdit'] ) && $settings['isEdit'] ) {
			$default = ( isset( $settings['default'] ) && isset( $settings['default']['aweber_tags'] ) ) ? $settings['default']['aweber_tags'] : '';
		}

		ob_start();

		ConvertPlugHelper::render_input_html(
			'aweber_tags',
			array(
				'class'   => 'cpro-tags',
				'type'    => 'text-wrap',
				'label'   => __( 'Tags', 'convertpro-addon' ),
				'help'    => __( 'Please separate tags with a comma.', 'convertpro-addon' ),
				'default' => $default,
			)
		);

		return ob_get_clean();
	}

	/**
	 * Mapping fields.
	 *
	 * @since 1.0.0
	 */
	public function render_mapping() {
		return self::$mapping_fields;
	}

	/**
	 * Get an instance of the 1.0 API.
	 *
	 * @since 1.5.9
	 * @param string $auth_code A valid API key.
	 * @return object The API instance.
	 */
	public function get_old_api( $auth_code ) {

		if ( $this->oldapi_instance ) {
			return $this->oldapi_instance;
		}

		if ( ! class_exists( 'AWeberAPI' ) ) {
			require_once CP_SERVICES_BASE_DIR . 'includes/vendor/aweber/aweber_api.php';
		}

		list( $auth_key, $auth_token, $req_key, $req_token, $oauth ) = explode( '|', $auth_code );

		$this->oldapi_instance                     = new AWeberAPI( $auth_key, $auth_token );
		$this->oldapi_instance->user->requestToken = $req_key;
		$this->oldapi_instance->user->tokenSecret  = $req_token;
		$this->oldapi_instance->user->verifier     = $oauth;

		return $this->oldapi_instance;
	}

	/**
	 * Check if 1.0 API is called.
	 *
	 * @since 1.5.9
	 * @param string $auth_code A valid API key.
	 * @return bool is true or false.
	 */
	public function is_old_api_called( $auth_code ) {
		return strpos( $auth_code, '|' ) !== false;
	}

	/**
	 * Subscribe an email address to AWeber.
	 *
	 * @since 1.0.0
	 * @param object $settings A module settings object.
	 * @param string $email The email to subscribe.
	 * @return array {
	 *      @type bool|string $error The error message or false if no error.
	 * }
	 */
	public function subscribe( $settings, $email ) {
		$account_data = ConvertPlugServices::get_account_data( $settings['api_connection'] );

		$response  = array(
			'error' => false,
		);
		$merge_arr = array();

		if ( ! $account_data ) {
			$response['error'] = __( 'There was an error subscribing to AWeber! The account is no longer connected.', 'convertpro-addon' );
		} else {
			$authentication_code = $account_data['auth_code'];

			// Different API calls for 1.0 and 2.0 API calls.
			if ( $this->is_old_api_called( $authentication_code ) ) {
				$api = $this->get_old_api( $authentication_code );
			} else {
				$api = $this->get_api( $authentication_code );
			}

			$data = array(
				'ws.op'           => 'create',
				'email'           => $email,
				'add_notes'       => $this->aweber_get_ip(),
				'ad_tracking'     => CPRO_BRANDING_NAME,
				'update_existing' => 'true',
			);

			if ( isset( $settings['aweber_tags'] ) ) {
				$data['tags'] = explode( ',', $settings['aweber_tags'] );
			}

			foreach ( $settings['param'] as $key => $p ) {
				if ( 'email' !== $key && 'date' !== $key ) {
					if ( 'custom_field' !== $settings['meta'][ $key ] ) {
						$merge_arr[ $settings['meta'][ $key ] ] = $p;
					} else {
						$merge_arr[ $settings['meta'][ $key . '-input' ] ] = $p;
					}
				}
			}

			if ( isset( $merge_arr['name'] ) ) {
				$data['name'] = $merge_arr['name'];
			}

			if ( ! empty( $merge_arr ) ) {
				$data['custom_fields'] = $merge_arr;
			}

			$aweber_list = $settings['aweber_list'];

			try {
				if ( $this->is_old_api_called( $authentication_code ) ) {
					$account = $api->getAccount( $account_data['access_token'], $account_data['access_secret'] );
				} else {
					$account = $api->getAccount( $authentication_code );
				}

				$list        = $account->loadFromUrl( $account->url . "/lists/{$aweber_list}" );
				$subscribers = $list->subscribers;
				$result      = $subscribers->create( $data );

			} catch ( AWeberAPIException $e ) {
				$err_message = $e->getMessage();
				if ( ! empty( $err_message ) && false !== strpos( $err_message, 'already subscribed' ) ) {
					$response['error'] = sprintf(
						/* translators: %s Error Message */
						__( 'There was an error subscribing to AWeber! %s', 'convertpro-addon' ),
						$e->getMessage()
					);
				}
			}
		}

		return $response;
	}

	/**
	 * Get User's IP
	 *
	 * @return string
	 * @since 1.0.0
	 */
	private function aweber_get_ip() {

		$hostname = gethostname();
		$ip       = gethostbyname( $hostname );

		return $ip;
	}
}
