<?php
if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}
if( ! class_exists( 'CPRO_Moosend' ) ) {
    class CPRO_Moosend {

        protected $apiUrl = 'https://api.moosend.com/v3/';

        protected $apiKey = null;

        public function __construct($apiKey = null)
        {
            $this->apiKey = $apiKey;
        }

        public function default_options()
        {
            return array(
                'apikey' => $this->apiKey
            );
        }

        public function make_request($action, $options = array(), $method = 'GET')
        {
            /* Build request options string. */
            $request_options = $this->default_options();

            $request_options = wp_parse_args($options, $request_options);

            $options_string = http_build_query($request_options);

            /* Build request URL. */
            $request_url = $this->apiUrl . $action . '?' . $options_string;

            /* Execute request based on method. */
            switch ($method) {
                case 'POST':
                    $request_url = $this->apiUrl . $action . '?apikey=' . $this->apiKey;
                    $args = [];
                    $args['body'] = $options;
                    $args['method'] = 'POST';
                    $args['headers'] = [
                        'Content-Type' => 'application/json',
                        'Accept'       => 'application/json'
                    ];
                    $response = wp_remote_post($request_url, $args);
                    break;

                case 'GET':
                    $response = wp_remote_get($request_url);
                    break;
            }

            /* If WP_Error, die. Otherwise, return decoded JSON. */
            if (is_wp_error($response)) {
                return [
                    'error'   => 'API_Error',
                    'message' => $response->get_error_message()
                ];
            } else {
                return json_decode($response['body'], true);
            }
        }

        /**
         * Test the provided API credentials.
         *
         * @access public
         * @return bool
         */
        public function connect()
        {
            $response = $this->make_request('lists.json', [
                'WithStatistics' => false,
                'ShortBy'        => 'CreatedOn',
                'SortMethod'     => 'ASC',
                'PageSize'       => 1
            ], 'GET');

            if (!empty($response['Error'])) {
                throw new \Exception( "This is not a valid API Key.", 1 );
            } else {
                return true;
            }
        }

        public function subscribe($data, $doubleOptIn = false)
        {

            $returnArray = array( 'error' => false );

            $url = $this->apiUrl . "subscribers/" . $data['MailingListId'] . "/subscribe.json?apikey=" . $this->apiKey;

            $moosendData = array(
                'Email' => $data['MemberEmail'], // Map MemberEmail to Email
                'HasExternalDoubleOptIn' => $doubleOptIn,
                'CustomFields' => array(), // Initialize CustomFields array
                'SubscribeType' => $data['SubscribeType']
            );

            // Set the name if available in $data
            if( isset( $data['Name'] ) ) {
                $moosendData['Name'] = $data['Name'];
            }

            if( ! empty( $data['CustomFields'] ) ) {
                foreach ( $data['CustomFields'] as $key => $value ){
                    // Check if value exists and is not empty
                    if( ! empty( $value ) ) {
                        $moosendData['CustomFields'][] = "$key=$value";
                    }
                }
            }

            $response = wp_remote_post( $url, [
                'method' => 'POST',
                'timeout' => 45,
                'blocking' => true,
                'header' => [
                    'Content-Type' => 'application/json',
                    'Accept'       => 'application/json'
                ],
                'body' => $moosendData
            ]);

            if(  ! is_wp_error( $response ) ) {
                $response_arr = json_decode( $response['body'] );

                if( isset( $response['response'] ) ) {
                    
                    $responseBody = json_decode($response['body'], true);

                    $code = isset( $response['response']['code'] ) ? $response['response']['code'] : false;
                    if ( isset($code) && $code !== 200 ) {
                        throw new \Exception( "It seems like the list has been changed or deleted.", 1 );    
                    }
                    if( !empty($responseBody['Error']) ) {
                        throw new \Exception( "Something went wrong!.", 1 );    
                    }
                }
            } else {
                $returnArray['error'] = true;            
            }

            return $returnArray;
        }


        /**
         * Get all Lists in the system.
         *
         * @access public
         * @return array
         */
        public function getList()
        {
            $response = $this->make_request('lists.json', [
                'WithStatistics' => false,
                'ShortBy'        => 'CreatedOn',
                'SortMethod'     => 'DESC',
                'PageSize'       => 999
            ], 'GET');

            if (empty($response['Error']) && !empty($response['Context']['MailingLists'])) {
                return $response['Context']['MailingLists'];
            }
            return false;
        }

        /**
         * Get single List in the system.
         *
         * @access public
         * @return array
         */
        public function getListDetails($listId)
        {
            $response = $this->make_request('lists/' . $listId . '/details.json', [
                'WithStatistics' => false
            ], 'GET');

            if (empty($response['Error'])) {
                return $response['Context'];
            }
            return false;
        }
    }
}
