<?php
if( ! class_exists( 'CPRO_Klaviyo' ) ) {
    class CPRO_Klaviyo {

        /**
         * The Base URL for the API.
         *
         * @since 1.0.0
         * @var string $root
         */
        public $root = 'https://a.klaviyo.com/api/';

        /**
         * The API Key.
         *
         * @since 1.0.0
         * @var string $apikey
         */
        public $apikey = null;

        /**
         * Constructor.
         *
         * @since 1.0.0
         * @var string $apikey
         */
        public function __construct( $apikey = null ) {
            
            if ( ! $apikey ) {
                throw new \Exception( "You must provide a Klaviyo API Key" , 1 );
            }

            $this->apikey = sanitize_text_field( $apikey );

        }

        /**
         * Get common header for any request.
         *
         * @since x.x.x
         * @var string $apikey
         */
        private function get_common_headers() {
            return [
                'Authorization' => 'Klaviyo-API-Key ' . $this->apikey,
                'accept' => 'application/json',
                'content-type' => 'application/json',
                'revision' => '2024-05-15',
            ];
        }

        /**
         * Make a connection call to Klaviyo.
         *
         * @since 1.0.0
         * @return array {
         *      @type bool|string $error The error message or false if no error.
         * }
         */
        public function connect() {

            $response = array( 'error' => false );

            $opts = array(
                'headers' => $this->get_common_headers(),
                'timeout' => 30,
                'httpversion' => '1.1'
            );

            $resp = wp_remote_get( $this->root . 'lists', $opts );

            if( ! is_wp_error( $resp ) ) {

                $body = wp_remote_retrieve_body( $resp );
                $request = json_decode( $body );

                if ( json_last_error() !== JSON_ERROR_NONE ) {
                    $response['error'] = __( 'Invalid JSON response.', 'convertpro-addon' );
                    return $response;
                }

                if( isset( $resp['response']['code'] ) ) {
                    if( $resp['response']['code'] != 200  ) {
                        // Not Connected
                        $response['error'] = isset($request->message) ? sanitize_text_field($request->message) : __( 'Error connecting to Klaviyo.', 'convertpro-addon' );
                    }
                } else {
                    // Not Connected
                    $response['error'] = __( 'Something went wrong.', 'convertpro-addon' );
                }

            } else {
                // Not Connected
                $response['error'] = sanitize_text_field( $resp->get_error_message() );
            }

            return $response;
        }
        

        /**
         * Get lists.
         *
         * @since 1.0.0
         * @return array
         * @type bool|string $error The error message or false if no error.
         *
         */
        public function getList() {
            $response = array('error' => false, 'lists' => array());
        
            $opts = array(
                'headers' => $this->get_common_headers(),
                'timeout' => 30,
                'httpversion' => '1.1'
            );
        
            $url = $this->root . 'lists/';
            $next_page_url = $url;
        
            do {
                $resp = wp_remote_get( $next_page_url, $opts );
        
                if ( ! is_wp_error( $resp ) ) {
                    $body = wp_remote_retrieve_body( $resp );
                    $request = json_decode( $body );
        
                    if ( json_last_error() !== JSON_ERROR_NONE ) {
                        throw new \Exception( __( 'Invalid JSON response.', 'convertpro-addon' ), 1 );
                    }
        
                    if ( isset( $resp['response']['code'] ) ) {
                        if ( $resp['response']['code'] != 200 ) {
                            // Handle error response.
                            throw new \Exception( isset( $request->message ) ? sanitize_text_field( $request->message ) : __( 'Error connecting to Klaviyo.', 'convertpro-addon' ), 1 );
                        } else {
                            // Merge the lists from this page into the final response.
                            $response['lists'] = array_merge( $response['lists'], $request->data );
        
                            // Check if there's a next page link.
                            if ( isset( $request->links->next  )) {
                                $next_page_url = $request->links->next;
                            } else {
                                $next_page_url = null; // No more pages.
                            }
                        }
                    } else {
                        // Handle unexpected response structure.
                        throw new \Exception( __( 'Something went wrong.', 'convertpro-addon' ), 1 );
                    }
                } else {
                    // Handle WP error.
                    throw new \Exception( sanitize_text_field( $resp->get_error_message() ), 1 );
                }
        
            } while ( $next_page_url ); // Continue looping until there's no next page.
        
            return $response;
        }

        /**
         * Subscribe an email address to Klaviyo.
         *
         * @since 1.0.0
         * @param int $list The list ID.
         * @param string $email The email to subscribe.
         * @param string $data The other data.
         * @return array {
         *      @type bool|string $error The error message or false if no error.
         * }
         */
        public function subscribe( $list, $email, $data ) {

            $data['api_key'] = $this->apikey;

            if( empty( $data['properties'] ) ) {
                unset( $data['properties'] );
            }

            // Prepare the profile data
            $profdata = [
                'type' => 'profile',
                'attributes' => [
                    'email' => sanitize_email( $email )
                ]
            ];            

            if ( isset( $data['properties'] ) ) {
                $profdata['attributes']['properties'] = $data['properties'];
                unset($data['properties']);
            }
            
            // Create the profile
            $profile_response = wp_remote_post($this->root . 'profiles', [
                'body' => json_encode(['data' => $profdata]),
                'headers' => $this->get_common_headers(),
                'sslverify' => true, // Enable SSL certificate verification
            ]);

            if (is_wp_error($profile_response)) {
                throw new \Exception( sanitize_text_field( $profile_response->get_error_message() ), 1 );
            }
        
            $profile_body = wp_remote_retrieve_body($profile_response);
            $profile_data = json_decode($profile_body, true);

            $response_code = wp_remote_retrieve_response_code($profile_response);
            
            if ( $response_code !== 201) {
                throw new \Exception( 'Unexpected response code: ' . $response_code, 1 );
            }

            if ( json_last_error() !== JSON_ERROR_NONE || !isset( $profile_data['data']['id'] ) ) {
                throw new \Exception( __( 'Failed to create or retrieve profile.', 'convertpro-addon' ), 1 );
            }

            $proid = $profile_data['data']['id'];

            $bodyOutput = wp_json_encode([
                'data' => [
                    'type' => 'profile-subscription-bulk-create-job',
                    'attributes' => [
                        'custom_source' => 'ConvertPro',
                        'profiles' => [
                            'data' => [
                                [
                                    'type' => 'profile',
                                    'id' => $proid,
                                    'attributes' => [
                                        'email' => sanitize_email( $email ),
                                        'subscriptions' => [
                                            'email' => [
                                                'marketing' => [
                                                    'consent' => 'SUBSCRIBED'
                                                ]
                                            ]
                                        ]
                                    ]
                                ]
                            ]
                        ]
                    ],
                    'relationships' => [
                        'list' => [
                            'data' => [
                                'type' => 'list',
                                'id' => sanitize_text_field( $list )
                            ]
                        ]
                    ]
                ]
            ]);

            // Subscribe the profile to the list
            $subscribe_response = wp_remote_post( $this->root . 'profile-subscription-bulk-create-jobs', [
                'body' => $bodyOutput,
                'headers' => $this->get_common_headers(),
                'sslverify' => true, // Enable SSL certificate verification.
            ]);

            if ( is_wp_error( $subscribe_response ) ) {
                throw new \Exception( sanitize_text_field( $subscribe_response->get_error_message() ), 1 );
            }

            $subscribe_body = wp_remote_retrieve_body($subscribe_response);
            $subscribe_data = json_decode($subscribe_body, true);

            // Check the HTTP response code
            $response_code = wp_remote_retrieve_response_code($subscribe_response);
            
            if ( $response_code !== 202 ) {
                $error_message = isset($subscribe_data['errors'][0]['detail']) ? sanitize_text_field($subscribe_data['errors'][0]['detail']) : __( 'Unknown error occurred.', 'convertpro-addon' );
                throw new \Exception( $error_message, 1 );
            }

            return ['error' => false];

        }

    }
}
