<?php
if ( ! class_exists( 'CPRO_Keap' ) ) {
    class CPRO_Keap {

        /**
         * The Base URL for the API.
         *
         * @since 1.0.0
         * @var string $root
         */
        public $root = 'https://api.infusionsoft.com/crm/rest/v1/';

        /**
         * The Access Token.
         *
         * @since 1.0.0
         * @var string $access_token
         */
        public $access_token = null;

        /**
         * Constructor.
         *
         * @since 1.0.0
         * @var string $access_token
         */
        public function __construct( $access_token = null ) {
            if ( ! $access_token ) {
                throw new \Exception( "You must provide a Keap (Infusionsoft) personal access token" , 1 );
            }

            $this->access_token = sanitize_text_field( $access_token );
        }

        /**
         * Get common headers for any request.
         *
         * @since 1.0.0
         * @var string $access_token
         */
        private function get_common_headers() {
            return [
                'X-Keap-API-Key' => $this->access_token,
                'accept' => 'application/json',
                'content-type' => 'application/json',
            ];
        }

        /**
         * Make a connection call to Keap.
         *
         * @since 1.0.0
         * @return array {
         *      @type bool|string $error The error message or false if no error.
         * }
         */
        public function connect() {

            $response = array( 'error' => false );

            $opts = array(
                'headers' => $this->get_common_headers(),
                'timeout' => 30,
                'httpversion' => '1.1'
            );

            $resp = wp_remote_get( $this->root . 'contacts', $opts );

            if ( ! is_wp_error( $resp ) ) {

                $body = wp_remote_retrieve_body( $resp );
                $request = json_decode( $body );

                if ( json_last_error() !== JSON_ERROR_NONE ) {
                    $response['error'] = __( 'Invalid JSON response.', 'convertpro-addon' );
                    return $response;
                }

                if ( isset( $resp['response']['code'] ) ) {
                    if ( $resp['response']['code'] != 200 ) {
                        // Not Connected
                        $response['error'] = isset( $request->message ) ? sanitize_text_field( $request->message ) : __( 'Error connecting to Keap (Infusionsoft).', 'convertpro-addon' );
                    }
                } else {
                    // Not Connected
                    $response['error'] = __( 'Something went wrong.', 'convertpro-addon' );
                }

            } else {
                // Not Connected
                $response['error'] = sanitize_text_field( $resp->get_error_message() );
            }

            return $response;
        }

        /**
         * Get lists.
         *
         * @since 1.0.0
         * @return array {
         *      @type bool|string $error The error message or false if no error.
         * }
         */
        public function getList() {

            $response = array( 'error' => false );

            $opts = array(
                'headers' => $this->get_common_headers(),
                'timeout' => 30,
                'httpversion' => '1.1'
            );

            $resp = wp_remote_get( $this->root . 'tags', $opts );

            if ( ! is_wp_error( $resp ) ) {
                $body = wp_remote_retrieve_body( $resp );

                $request = json_decode( $body );

                if ( json_last_error() !== JSON_ERROR_NONE ) {
                    throw new \Exception( __( 'Invalid JSON response.', 'convertpro-addon' ), 1 );
                }

                if ( isset( $resp['response']['code'] ) ) {
                    if ( $resp['response']['code'] != 200 ) {
                        // Not Connected
                        throw new \Exception( isset( $request->message ) ? sanitize_text_field( $request->message ) : __( 'Error connecting to Keap (Infusionsoft).', 'convertpro-addon' ), 1 );
                    } else {
                        $response['lists'] = $request->tags;
                    }
                } else {
                    // Not Connected
                    throw new \Exception( __( 'Something went wrong.', 'convertpro-addon' ), 1 );
                }

            } else {
                // Not Connected
                throw new \Exception( sanitize_text_field( $resp->get_error_message() ), 1 );
            }

            return $response;
        }

        /**
         * Create custom field in Keap.
         *
         * @since x.x.x
         * @param string $label The custom field label.
         */
        public function create_custom_field( $label ) {

            $label = sanitize_text_field($label);

            // Retrieve existing custom fields.
            $response = wp_remote_get($this->root . 'contacts/model', [
                'headers' => $this->get_common_headers(),
                'sslverify' => true,
            ]);

            if (is_wp_error($response)) {
                throw new \Exception(sanitize_text_field($response->get_error_message()), 1);
            }

            $body = wp_remote_retrieve_body($response);
            $data = json_decode($body, true);

            // Check if custom field already exists
            if (isset($data['custom_fields']) && is_array($data['custom_fields'])) {
                foreach ($data['custom_fields'] as $field) {
                    if ($field['label'] === $label) {
                        return $field['id'];
                    }
                }
            }

            $field_data = [
                'label' => $label,
                'field_type' => 'Text', // Adjust this based on the type of custom field you want
            ];

            $response = wp_remote_post($this->root . 'contacts/model/customFields', [
                'body' => json_encode($field_data),
                'headers' => $this->get_common_headers(),
                'sslverify' => true,
            ]);
        
            if (is_wp_error($response)) {
                throw new \Exception(sanitize_text_field($response->get_error_message()), 1);
            }
        
            $body = wp_remote_retrieve_body($response);
            $data = json_decode($body, true);
        
            if (isset($data['id'])) {
                return $data['id'];
            } else {
                throw new \Exception(__('Failed to create custom field.', 'convertpro-addon'), 1);
            }
        }
        

        /**
         * Subscribe an email address to Keap(Infusionsoft).
         *
         * @since 1.0.0
         * @param int $list The list ID.
         * @param string $email The email to subscribe.
         * @param string $data The other data.
         * @return array {
         *      @type bool|string $error The error message or false if no error.
         * }
         */
        public function subscribe( $list, $email, $data ) {

            $contact_id = '';

            $data['access_token'] = $this->access_token;
        
            if ( empty( $data['properties'] ) ) {
                unset( $data['properties'] );
            }
        
            // Prepare the contact data
            $contact_data = [
                'email_addresses' => [
                    [
                        'email' => sanitize_email( $email ),
                        'field' => 'EMAIL1'
                    ]
                ],                    
                'opt_in_reason' => __( 'User opted in via form submission', 'convertpro-addon' ),
            ];

            // Known fields mapping.
            $known_fields = ['given_name', 'family_name', 'email', 'phone_number', 'address', 'zip_code', 'website'];
            
            if( isset($data['properties']) && is_array($data['properties']) ) {

                // Initialize custom fields array.
                $custom_fields = [];

                foreach ($data['properties'] as $key => $value) {
                    $key = sanitize_text_field($key);
                    $value = sanitize_text_field($value);
                    
                    if (in_array($key, $known_fields)) {
                        // Map known fields
                        if ($key === 'phone_number') {
                            $contact_data['phone_numbers'][] = [
                                'number' => $value,
                                'field' => 'PHONE1'
                            ];
                        } elseif ($key === 'address') {
                            if ( isset( $contact_data['addresses'] ) ) {
                                $contact_data['addresses'][0]['line1'] = $value;
                            } else {
                                $contact_data['addresses'][] = [
                                    'field' => 'BILLING',
                                    'line1' => $value // Adjust this according to your actual data structure
                                ];
                            }
                        } elseif ($key === 'zip_code') {
                            if ( isset( $contact_data['addresses'] ) ) {
                                $contact_data['addresses'][0]['zip_code'] = $value;
                            } else {
                                $contact_data['addresses'][] = [
                                    'field' => 'BILLING',
                                    'zip_code' => $value // Adjust this according to your actual data structure
                                ];
                            }
                        } else {
                            $contact_data[$key] = $value;
                        }
                    } else {
                        // Create custom field if not exist
                        $field_id = $this->create_custom_field($key);
                        $custom_fields[] = [
                            'content' => $value,
                            'id' => $field_id
                        ];
                    }
                }

                // Add custom fields to contact data if any.
                if ( ! empty( $custom_fields ) ) {
                    $contact_data['custom_fields'] = $custom_fields;
                }
            }

            // Step 1: Search for existing contact by email.
            $search_response = wp_remote_get( $this->root . 'contacts?email=' . urlencode( sanitize_email( $email ) ), [
                'headers' => $this->get_common_headers(),
                'sslverify' => true,
            ]);

            if ( is_wp_error( $search_response ) ) {
                throw new \Exception( sanitize_text_field( $search_response->get_error_message() ), 1 );
            }

            $search_body = wp_remote_retrieve_body( $search_response );
            $search_data = json_decode( $search_body, true );
            
            if ( !empty( $search_data['contacts'] ) && isset( $search_data['contacts'][0]['id'] ) ) {
                // Contact exists, update the existing contact
                $contact_id = $search_data['contacts'][0]['id'];
            } else {
        
                // Create or update the contact
                $contact_response = wp_remote_post( $this->root . 'contacts', [
                    'body' => json_encode( $contact_data ),
                    'headers' => $this->get_common_headers(),
                    'sslverify' => true, // Enable SSL certificate verification.
                ] );
            
                if ( is_wp_error( $contact_response ) ) {
                    throw new \Exception( sanitize_text_field( $contact_response->get_error_message() ), 1 );
                }
            
                $contact_body = wp_remote_retrieve_body( $contact_response );
                $contact_data = json_decode( $contact_body, true );
            
                // Check the HTTP response code
                $response_code = wp_remote_retrieve_response_code( $contact_response );
            
                if ( $response_code !== 200 && $response_code !== 201 ) {
                    $error_message = isset( $contact_data['message'] ) ? sanitize_text_field( $contact_data['message'] ) : __( 'Unknown error occurred.', 'convertpro-addon' );
                    throw new \Exception( $error_message, 1 );
                }
            
                // Get the contact ID from the response
                $contact_id = $contact_data['id'];
            }
        
            // Add tags to the contact
            foreach ( $list as $tag_id ) {
                $tag_response = wp_remote_post( $this->root . 'contacts/' . $contact_id . '/tags', [
                    'body' => json_encode( ['tagIds' => [$tag_id]] ),
                    'headers' => $this->get_common_headers(),
                    'sslverify' => true,
                ] );
        
                if ( is_wp_error( $tag_response ) ) {
                    throw new \Exception( sanitize_text_field( $tag_response->get_error_message() ), 1 );
                }
        
                $tag_body = wp_remote_retrieve_body( $tag_response );
                $tag_data = json_decode( $tag_body, true );
        
                // Check the HTTP response code for tag addition
                $tag_response_code = wp_remote_retrieve_response_code( $tag_response );
        
                if ( $tag_response_code !== 200 && $tag_response_code !== 201 ) {
                    $tag_error_message = isset( $tag_data['message'] ) ? sanitize_text_field( $tag_data['message'] ) : __( 'Unknown error occurred.', 'convertpro-addon' );
                    throw new \Exception( $tag_error_message, 1 );
                }
            }
        
            return [ 'error' => false ];
        }
        
    }
}
