<?php
/**
 * Convert Pro Addon AJAX Handler
 *
 * @package Convert Pro Addon
 */

if ( ! class_exists( 'CP_Automation_Handler' ) ) {

	/**
	 * Automation handler class.
	 */
	class CP_Automation_Handler {

		/**
		 * Constructor of the file.
		 */
		public function __construct() {
			add_action( 'wp_ajax_cp_install_and_activate', array( $this, 'install_and_activate_plugin' ) );
			add_action( 'wp_ajax_cp_activate_plugin', array( $this, 'activate_plugin' ) );
			add_action( 'wp_ajax_cp_check_suretriggers_connection', array( $this, 'check_suretriggers_connection' ) );
		}

		/**
		 * Verify nonce
		 *
		 * @param string $nonce_action nonce verification.
		 * @return void
		 */
		private function verify_nonce( $nonce_action ) {
			if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( $_POST['nonce'], $nonce_action ) ) {
				wp_send_json_error( __( 'Invalid nonce. Action not allowed.', 'convertpro-addon' ) );
			}
		}

		/**
		 * Check if the user is connected to SureTriggers.
		 */
		public function check_suretriggers_connection() {

			// Validate the nonce.
			$this->verify_nonce( 'cp_suretriggers' );

			$is_connected = apply_filters( 'suretriggers_is_user_connected', '' );

			if ( $is_connected ) {
				wp_send_json_success( array( 'authenticated' => true ) );
			} else {
				wp_send_json_error( array( 'authenticated' => false ) );
			}
		}

		/**
		 * Check user permissions
		 *
		 * @param string $capability check permissions.
		 * @return void
		 */
		private function check_permissions( $capability ) {
			if ( ! current_user_can( $capability ) ) {
				wp_send_json_error( __( 'You do not have permission to perform this action.', 'convertpro-addon' ) );
			}
		}

		/**
		 * Install and activate the SureTriggers plugin
		 *
		 * @return void
		 */
		public function install_and_activate_plugin() {

			$this->verify_nonce( 'cp_suretriggers' );
			$this->check_permissions( 'install_plugins' );

			// Include necessary WordPress plugin installation files.
			include_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';
			include_once ABSPATH . 'wp-admin/includes/plugin-install.php';

			// Get plugin information from the WordPress plugin repository.
			$plugin_slug = 'suretriggers';
			$plugin_info = plugins_api( 'plugin_information', array( 'slug' => $plugin_slug ) );

			// Suppress output during plugin installation.
			$installed = $this->install_plugin( $plugin_info->download_link );

			wp_cache_flush();

			if ( ! is_wp_error( $installed ) && $installed ) {
				$plugin_slug = 'suretriggers/suretriggers.php';

				if ( ! is_plugin_active( $plugin_slug ) ) {
					$result = activate_plugin( $plugin_slug );
					delete_transient( 'st-redirect-after-activation' );

					if ( is_wp_error( $result ) ) {
						wp_send_json_error( __( 'Could not install SureTriggers.', 'convertpro-addon' ) );
					}

					wp_send_json_success( __( 'Plugin activated successfully.', 'convertpro-addon' ) );
				}
				wp_send_json_success( __( 'SureTriggers activated.', 'convertpro-addon' ) );
			} else {
				wp_send_json_error( __( 'Could not install SureTriggers.', 'convertpro-addon' ) );
			}
		}

		/**
		 * Install the plugin file
		 *
		 * @param string $link install plugin.
		 */
		public function install_plugin( $link ) {
			wp_cache_flush();

			$upgrader  = new Plugin_Upgrader( new WP_Ajax_Upgrader_Skin() );
			$installed = $upgrader->install( $link );

			return $installed;
		}


		/**
		 * Activate the plugin
		 *
		 * @return void
		 */
		public function activate_plugin() {

			$this->verify_nonce( 'cp_suretriggers' );

			$this->check_permissions( 'activate_plugins' );

			$plugin_slug = 'suretriggers/suretriggers.php';

			if ( ! is_plugin_active( $plugin_slug ) ) {
				$result = activate_plugin( $plugin_slug );
				delete_transient( 'st-redirect-after-activation' );

				if ( is_wp_error( $result ) ) {
					wp_send_json_error( $result->get_error_message() );
				}

				wp_send_json_success( __( 'Plugin activated successfully.', 'convertpro-addon' ) );
			}

			wp_send_json_success( __( 'Plugin is already activated.', 'convertpro-addon' ) );
		}
	}

	new CP_Automation_Handler();
}
